function [errorL2,errorH1]=errorNS2D(nov_p,vertices,noe,elements,aree,uh,ph,fem,t,varargin)
%ERRORNS2D error function for the Navier-Stokes problem.
%   [ERRORL2]=ERRORNS2D(NOVP,VERTICES,NOE,ELEMENTS,AREE,UH,PH,FEM,T) computes
%   the L^2 error at time T for FEM finite elements solutions UH, VH and PH of the Navier-Stokes
%   equantions. The exact solutions must be precised in the functions NSSOLU2D, NSSOLV2D
%   and NSSOLP2D in the user's directory. To compute the integral a quadrature formula of order 10 
%   is used. AREE is the vector of the areas of the triangles of the mesh.
%
%   [ERRORL2,ERRORH1]=ERRORNS2D(NOVP,VERTICES,NOE,ELEMENTS,AREE,UH,PH,FEM,T) 
%   computes the L^2 and the H^1 errors. In this case the functions NSSOLU2D, NSSOLV2D
%   and NSSOLP2D must provide also the values of the first derivatives of the analytical solutions.
%
%   [ERRORL2,ERRORH1]=ERRORNS2D(NOVP,VERTICES,NOE,ELEMENTS,AREE,UH,PH,FEM,T,P1,P2,...) 
%   passes parameters P1,P2,... to the functions NSSOLU2D, NSSOLV2D and NSSOLP2D.

%       F. Saleri 20-05-03.

z = 0;

[nln,nbn]=select(fem);
[d,nov]=size(vertices);

errorL2 = [0; 0; 0];

[csi,eta,w,phi,dxphi,dyphi]=basis_on_quad2D(fem,10);
[csi,eta,w,phip,dxphip,dyphip]=basis_on_quad2D('P1',10);
eta1 = 1-csi-eta;

if nargout == 1  % only L^2 errors are computed
    for ie = 1:noe
        % Vertices
        i = elements(1,ie);   j = elements(2,ie);    k = elements(3,ie); 
        a1x = vertices(1,i);  a2x = vertices(1,j);   a3x = vertices(1,k);
        a1y = vertices(2,i);  a2y = vertices(2,j);   a3y = vertices(2,k);
        x = eta1*a1x + csi*a2x + eta*a3x;
        y = eta1*a1y + csi*a2y + eta*a3y;
        e = elements(1:nln,ie);
        if fem == 'Pc'
            ep = ie;
        else
            ep = e(1:3);
        end
        uhq = uh(e)'*phi;
        vhq = uh(e+nov)'*phi;
        phq = ph(ep)'*phip;
        uq = feval('nssolu2D',x,y,t,varargin{:});
        vq = feval('nssolv2D',x,y,t,varargin{:});
        pq = feval('nssolp2D',x,y,t,varargin{:});
        errorL2(1) = errorL2(1) + sum(w.*((uhq-uq).^2))*2*aree(ie);
        errorL2(2) = errorL2(2) + sum(w.*((vhq-vq).^2))*2*aree(ie);
        errorL2(3) = errorL2(3) + sum(w.*((phq-pq).^2))*2*aree(ie);
    end
elseif nargout == 2
    [detjac,dcdx,dcdy,dedx,dedy]=geotrasf2D(nov,vertices,noe,elements);
    errorH1 = [0; 0; 0];
    for ie = 1:noe
        % Vertices
        i = elements(1,ie);   j = elements(2,ie);    k = elements(3,ie); 
        a1x = vertices(1,i);  a2x = vertices(1,j);   a3x = vertices(1,k);
        a1y = vertices(2,i);  a2y = vertices(2,j);   a3y = vertices(2,k);
        x = eta1*a1x + csi*a2x + eta*a3x;
        y = eta1*a1y + csi*a2y + eta*a3y;
        e = elements(1:nln,ie);
        if fem == 'Pc'
            ep = ie;
        else
            ep = e(1:3);
        end
        uhq = uh(e)'*phi;
        vhq = uh(e+nov)'*phi;
        phq = ph(ep)'*phip;
        uhdxq = uh(e)'*(dcdx(ie)*dxphi+dedx(ie)*dyphi);
        uhdyq = uh(e)'*(dcdy(ie)*dxphi+dedy(ie)*dyphi);
        vhdxq = uh(e+nov)'*(dcdx(ie)*dxphi+dedx(ie)*dyphi);
        vhdyq = uh(e+nov)'*(dcdy(ie)*dxphi+dedy(ie)*dyphi);
        phdxq = ph(ep)'*(dcdx(ie)*dxphip+dedx(ie)*dyphip);
        phdyq = ph(ep)'*(dcdy(ie)*dxphip+dedy(ie)*dyphip);
        [uq,uxq,uyq] = feval('nssolu2D',x,y,t,varargin{:});
        [vq,vxq,vyq] = feval('nssolv2D',x,y,t,varargin{:});
        [pq,pxq,pyq] = feval('nssolp2D',x,y,t,varargin{:});
        errorL2(1) = errorL2(1) + sum(w.*((uhq-uq).^2))*2*aree(ie);
        errorL2(2) = errorL2(2) + sum(w.*((vhq-vq).^2))*2*aree(ie);
        errorL2(3) = errorL2(3) + sum(w.*((phq-pq).^2))*2*aree(ie);
        errorH1(1) = errorH1(1) + sum(w.*((uhdxq-uxq).^2+(uhdyq-uyq).^2))*2*aree(ie);
        errorH1(2) = errorH1(2) + sum(w.*((vhdxq-vxq).^2+(vhdyq-vyq).^2))*2*aree(ie);
        errorH1(3) = errorH1(3) + sum(w.*((phdxq-pxq).^2+(phdyq-pyq).^2))*2*aree(ie);
    end
    errorH1 = errorH1 + errorL2;
    errorH1 = sqrt(errorH1);
end
errorL2 = sqrt(errorL2);

return